/*	PiSyn.c - Edit 4

	LoadICE Version 2.4
	Copyright (C) 1990-94 Grammar Engine, Inc.
	All rights reserved
	
	NOTICE:  This software source is a licensed copy of Grammar Engine's
	property.  It is supplied to you as part of support and maintenance
	of some Grammar Engine products that you may have purchased.  Use of
	this software is strictly limited to use with such products.  Any
	other use constitutes a violation of this license to you.
*/

/*	 - Promice Syntax Parser
*/
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "piconfig.h"
#include "pistruct.h"
#include "pierror.h"
#include "pisyn.h"

#ifdef ANSI
static void psgets(void);
static void pschar(void);
static void psline(void);
static void psnum(void);
static void psnuml(void);
static void psnums(void);
static void psid(void);
static short psbreak(void);
static long psrdx(void);

#else
static void psgets();
static void pschar();
static void psline();
static void psnum();
static void psnuml();
static void psnums();
static void psid();
static short psbreak();
static long psrdx();
#endif

/*	local 'global' variables -
	- used by the routines in this module
*/
static char *ps_brk = " ,:=[](){}\n\r\t\""; /* break characters */
static char *psi;		/* current input pointer */
static short ps_nun;	/* no input read or found */
static short ps_rad;	/* radix for numeric input */
static short ps_end;	/* end of input seen */
static char ps_syn;	/* current script character */
static char ps_string[PIC_FN];	/* place to hold string pointed to by ps_str */

/*
	The parser will search the script table (**pscript) for entry matching
	the command portion of the input string. A match is made if all characters
	match till the '&' is incountered in the script. Then on the input is
	parsed as indicated by the charcters below:

	$	-string
	*	-hex number
	#	-decimal number
	@	-list of hex numbers
	&	-list of decimal numbers
	%	-number imbedded in a string (ROM part# etc.)
	:	-unit id (unless reading a string - i.e. filename)
	+	-a single character
	!	-rest of the input line
	(	-call function in (*psynf[x]()) where x=letter follwing '('

	The parser initializes globals to be used by functions in PISYNF.c
*/

void pisyn(char *is, char **pscript, void (*psynf[])(void))
{
	char *sp;
	char **ts;
	short i;
	char c;
	
	if (pxdisp&PXMH) { /* diagnostic output */
		i = strlen(is);
		c = is[i-1];
		if (c == '\n' || c == '\r')
			is[i-1] = '\0';
		printf("parsing `%s`\n",is);
	}

	pi_estr1 = is;			/* error string - when error */
	pi_eloc = (char *)0;	/* possible pointer to error location */
	
	/* while not the end of input or no error */

	while (*is != '\0' && *is != '\n' && *is != '\r' && !pxerror) {
		if ((*is == ' ') || (*is == '\t')) {
			is++;		/* skip over white spaces in front on input */
			continue;
		}

		/* initialize globals */

		pi_eloc = is;
		ps_str = (char *)0;
		ps_emb = -1;
		ps_end = 0;
		ps_nun = 0;
		ps_id = -1;
		ps_sli = 0;
		ps_li = 0;
		ps_i = 0;

		ts = pscript;

		/* search till end of table or match */

		while ((sp = *(ts++)) != (char *)0) {
			psi = is;
			while (*psi == *sp) {
				psi++;
				sp++;
			}
			if (*sp != '&')		/* if not '&' then no match */
				if (*sp != '~')	/* unless default action wanted */
					continue;
			pi_estr2 = sp;

			/* parse further input till '(' encountered or error */

			while ((*(++sp) != '(') && (!pxerror)) {
				ps_rad = 10;
				ps_syn = *sp;
				switch(*sp) {
					case '$':		/* read a string from input */
						psgets();
						break;
					case '+':		/* a single character */
						pschar();
						break;
					case '*':		/* a hex number */
						ps_rad = 16;
						psnum();
						break;
					case '#':		/* a decimal number */
						psnum();
						break;
					case '@':		/* list of hex numbers */
						ps_rad = 16;
						psnuml();
						break;
					case '&':		/* list of decimal numbers */
						psnuml();
						break;
					case '%':		/* embedded number */
						psnums();
						break;
					case ':':		/* possible unit ID */
						psid();
						break;
					case '!':		/* rest of the line */
						psline();
						break;
					case '\0':		/* end of input */
						break;
					default:		/* bad news */
						pxerror = PGE_SYN;
						break;
				}
			}

			if (!pxerror) {		/* if all is kool */
				c = *(++sp);	/* compute function index in table */
				if (isalpha(c)) {
					if (islower(c))
						c -= 'a';
					else {
						c -= 'A';
						c += 26;
					}

					/* more diagnostic output */
					if (pxdisp & PXMH) {
						if (ps_str != (char *)0)
							printf(" ps_str=`%s`",ps_str);
						if (ps_emb >= 0)
							printf(" ps_emb=%ld",ps_emb);
						if (ps_id >= 0)
							printf(" ps_id=%d",ps_id);
						if (ps_i) {
							printf(" Nums(%d):",ps_i);
							for (i=0; i<ps_i; i++)
								printf(" %ld",ps_num[i]);
						}
						if (ps_li) {
							printf(" List(%d/%d):",ps_li,ps_sli);
							for (i=0; i<ps_li; i++)
								printf(" %ld",ps_numl[i]);
						}
						printf("\n");
					}

					/* This is where the action is. *****************
					 * The table is at the end of pisynf.c
					 * Note that pishell.c calls pisyn() with possibly
					 * 3 different parse tables,
					 *  (these are in piscript.h)
					 * but they all index the same table of functions.
					 * Any hacking will have to take care to keep all
					 * four of these things in sync.
					 */
					(*psynf[c])();	
				}
			}
			break;
		} /* END of inner while loop */

		if (sp == (char *)0)	/* if end of script table */
			if (!pxerror)		/* and no error then */
				pxerror = PGE_CMD;	/* command not found */

		is = psi;		/* else keep processing input */
		if (piflags&PiiX)
			break;
	} /* END of outer while loop */
}


/* `psgets` - read a string from the input */

static void psgets ( void )
{
	/* skip break charcters */
	while (psbreak() & !ps_end)
		psi++;

	/* nothing to read */
	if (ps_end) {
		ps_str = (char *)0;
		return;
	}

	/* if ran into next arg (in command line) */
	if (*psi == '-') {
		pxerror = PGE_INP;
		return;
	}

	ps_str = psi;
	if (ps_bchr == '\"')	/* read everyting in "" */
		while (*psi != '\"')
		    psi++;
	else
		while (!psbreak())
			psi++;

	if (*psi != '\0') {
		*psi = '\0';
		strcpy(ps_string,ps_str);
		*psi = ' ';
		ps_str = ps_string;
	}
}

/* `pschar` - read a single character */

static void pschar()
	{
	while (psbreak() & !ps_end)
		psi++;

	if (ps_end)
		ps_char = '\0';
	else
		ps_char = *psi++;
	}

/* `psline` - read a rest of the input line */

static void psline()
	{
	while (psbreak() & !ps_end)
		psi++;

	if (ps_end)
		{
		ps_str = (char *)0;
		}
	else
		ps_str = psi;
	psi += strlen(ps_str);
	}

/* `psnum` - read a number from the input */

static void psnum()
	{
	ps_num[ps_i] = psrdx();
	if (!ps_nun)
		if (++ps_i >= PIC_NUM)
			pxerror = PGE_NUM;
	}

/* `psnuml` - read a list of zero or more numbers from the input */

static void psnuml()
	{
	while (ps_li < PIC_LST)
		{
		if ((ps_bchr == '(') || (ps_bchr == ')'))
			{
			ps_rad = 16;
			ps_sli++;
			}
		ps_numl[ps_li++] = psrdx();
		if (ps_nun)
			{
			ps_li--;
			ps_li -= ps_sli;
			return;
			}
		}
	pxerror = PGE_LST;
	}

/* `psnums` - read an embedded number from the input */

static void psnums()
	{
	char c;

	while (psbreak() & !ps_end)
		psi++;
	if ((*psi == '-') || ps_end) /* we hit the next arg */
		return;
	while (!psbreak())
		{
		c = *(psi++);
		if (c < '0' || c > '9')
			continue;
		if (ps_emb == -1)
			ps_emb = 0;
		ps_emb = ps_emb * 10 + (c - '0');
		}
	}

/* `psid` - read a unit ID from the input */

static void psid()
	{
	char *tp;
	
	tp = psi;
	
	ps_id = (short)psrdx();
	
	if (ps_nun || ps_bchr != ':' || ps_end)
		{
		ps_nun = 0;
		ps_end = 0;
		ps_id = -1;
		psi = tp;
		}
	else
		{
		if (ps_id >= PIC_NR)
			{
			ps_id = -1;
			pxerror = PGE_BIG;
			}
		}
	}

/* `psrdx` - read string as hex */

static long psrdx()
	{
	char c;
	char *tis;
	long value = 0;
	short neg = 0;
	
	while (psbreak() && !ps_end)
		psi++;
	if (ps_bchr == '(')
		{
		ps_rad = 16;
		ps_sli++;
		}
	if (ps_end || (*psi == '-'))
		{
		ps_nun = 1;
		return(0);
		}

	tis = psi;
	while (!psbreak())
		{
		c = *(psi++);
		if (c == 'x' || c == 'X')
			{
			ps_rad = 16;
			continue;
			}
		if (c == '-')
			{
			neg = 1;
			continue;
			}
		if (ps_rad == 16)
			{
			if (c >= '0' && c <= '9')
				{	
				value = value * 16 + (long)(c - '0');
				continue;
				}
			if (c >= 'A' && c <= 'F')
				{
				value = value * 16 + (long)(c - 'A' + 10);
				continue;
				}
			if (c >= 'a' && c <= 'f')
				{
				value = value * 16 + (long)(c - 'a' + 10);
				continue;
				}
			else
				{
				ps_nun = 1;
				break;
				}
			}
		else
			{
			if (c >= '0' && c <= '9')
				{
				value = value * 10 + (long)(c - '0');
				continue;
				}
			else
				{
				ps_nun = 1;
				break;
				}
			}
		}
	if (ps_nun)
		psi = tis;

	if (neg)
		return(-value);

	return(value);
	}

/* `psbreak` - returns non-zero if break char */

static short
psbreak ( void )
{
	char *bs = ps_brk;

	/* tjt - 9-15-2012
	 * This code was segfaulting when trying to write
	 * a null into a constant string (and when there
	 * already was a null there).  Old C compilers
	 * allowed this, but cannot do this these days.
	 * I reordered the code a bit, the assumption being
	 * that a constant string won't contain \n or \r
	 * This seems to fix things.
	 */
	if ( *psi == '\n' || *psi == '\r' )
	    *psi = '\0';

	if ( *psi == '\0' ) {
	    ps_bchr = '\0';
	    ps_end = 1;
	    return (1);
	}

#ifdef notdef
	if (*psi == '\0' || *psi == '\n' || *psi == '\r') {
		ps_bchr = *psi = '\0';
		ps_bchr = '\0';
		ps_end = 1;
		return(1);
	}
#endif

	while (*bs != '\0') {
	    if (*bs == *psi) {
		if ((ps_syn != '$') || (*bs != ':')) {
		    ps_bchr = *bs;
		    return(1);
		}
	    }
	    bs++;
	}

	return (0);
}

/* THE END */
