/* --------------------------------------------------------------------------
  FILE        : pru.c 				                             	 	        
  PROJECT     : DA8xx/OMAP-L138/C674x PRU Development
  DESC        : PRU Load and Run APIs
  VERSION     : 1.0   
-------------------------------------------------------------------------- */ 

/************************************************************
* Include Files                                             *
************************************************************/

// General type include
#include "tistdtypes.h"

// CSL header files
#include "csl/cslr.h"
#include "csl/soc_OMAPL138.h"
#include "csl/cslr_prucore.h"
#include "csl/cslr_psc_OMAPL138.h"

// This module's header file
#include "pru.h"


/************************************************************
* Explicit External Declarations                            *
************************************************************/


/************************************************************
* Local Macro Declarations                                  *
************************************************************/


/************************************************************
* Local Typedef Declarations                                *
************************************************************/


/************************************************************
* Local Function Declarations                               *
************************************************************/
static void LOCAL_pruPscDisable( void );
static void LOCAL_pruPscEnable( void );


/************************************************************
* Local Variable Definitions                                *
************************************************************/


/************************************************************
* Global Variable Definitions                               *
************************************************************/


/************************************************************
* Global Function Definitions                               *
************************************************************/

// Load the specified PRU with code
Uint32 PRU_load (Uint8 pruNum, Uint32* pruCode, Uint32 codeSizeInWords)
{ 
  Uint32 *pruIram; 
  Uint32 i;
  
  if (pruNum == CSL_PRUCORE_0)
  {
    pruIram = (Uint32 *) PRU0_PROG_RAM_START;
  }
  else if (pruNum == CSL_PRUCORE_1)
  {
    pruIram = (Uint32 *) PRU1_PROG_RAM_START;
  }
  else
  {
    return E_FAIL;
  }
  
  PRU_enable();

  // Copy dMAX code to its instruction RAM
  for(i=0; i<codeSizeInWords; i++)
  {
    pruIram[i] = pruCode[i];
  }

  return E_PASS;

}

Uint32 PRU_run (Uint8 pruNum)
{
  CSL_PrucoreRegsOvly hPru;

  if (pruNum == CSL_PRUCORE_0)
  {
    hPru = (CSL_PrucoreRegsOvly) CSL_PRUCORE_0_REGS;
  }
  else if (pruNum == CSL_PRUCORE_1)
  {
    hPru = (CSL_PrucoreRegsOvly) CSL_PRUCORE_1_REGS;
  }
  else
  {
    return E_FAIL;
  }

  // Enable dMAX, let it execute the code we just copied
  CSL_FINST(hPru->CONTROL,PRUCORE_CONTROL_COUNTENABLE,ENABLE);
  CSL_FINST(hPru->CONTROL,PRUCORE_CONTROL_ENABLE,ENABLE);
  return E_PASS;  
}

Uint32 PRU_waitForHalt (Uint8 pruNum, Int32 timeout)
{
  CSL_PrucoreRegsOvly hPru;

  Int32 cnt = timeout;

  if (pruNum == CSL_PRUCORE_0)
  {
    hPru = (CSL_PrucoreRegsOvly) CSL_PRUCORE_0_REGS;
  }
  else if (pruNum == CSL_PRUCORE_1)
  {
    hPru = (CSL_PrucoreRegsOvly) CSL_PRUCORE_1_REGS;
  }
  else
  {
    return E_FAIL;
  }  

  while( CSL_FEXT(hPru->CONTROL,PRUCORE_CONTROL_RUNSTATE) == CSL_PRUCORE_CONTROL_RUNSTATE_RUN )
  {
    if (cnt>0) 
    {  
      cnt--;
    }
    if (cnt == 0)
    {
      return E_TIMEOUT;
    }
  }
  
  return E_PASS;
}

Uint32 PRU_disable( void )
{
  CSL_PrucoreRegsOvly hPru;

  // Disable PRU0
  hPru = (CSL_PrucoreRegsOvly) CSL_PRUCORE_0_REGS;  
  CSL_FINST(hPru->CONTROL,PRUCORE_CONTROL_COUNTENABLE,DISABLE);
  CSL_FINST(hPru->CONTROL,PRUCORE_CONTROL_ENABLE,DISABLE);

  // Reset PRU0
  hPru->CONTROL = CSL_PRUCORE_CONTROL_RESETVAL;
  
  // Disable PRU1
  hPru = (CSL_PrucoreRegsOvly) CSL_PRUCORE_1_REGS;  
  CSL_FINST(hPru->CONTROL,PRUCORE_CONTROL_COUNTENABLE,DISABLE);
  CSL_FINST(hPru->CONTROL,PRUCORE_CONTROL_ENABLE,DISABLE);
  
  // Reset PRU1
  hPru->CONTROL = CSL_PRUCORE_CONTROL_RESETVAL;
  
  // Disable PRU SS
  LOCAL_pruPscDisable();

  return E_PASS;  
}

Uint32 PRU_enable ( void )
{
  CSL_PrucoreRegsOvly hPru;
  
  // Enable PRU SS
  LOCAL_pruPscEnable();
  
  // Reset PRU0
  hPru = (CSL_PrucoreRegsOvly) CSL_PRUCORE_0_REGS;
  hPru->CONTROL = CSL_PRUCORE_CONTROL_RESETVAL;
  
  // Reset PRU1
  hPru = (CSL_PrucoreRegsOvly) CSL_PRUCORE_1_REGS;
  hPru->CONTROL = CSL_PRUCORE_CONTROL_RESETVAL;  

  return E_PASS;    
}


/************************************************************
* Local Function Definitions                                *
************************************************************/

static void LOCAL_pruPscEnable( void )
{
  CSL_PscRegsOvly PSC = (CSL_PscRegsOvly) CSL_PSC_0_REGS;
  
  // Wait for any outstanding transition to complete
  while ( CSL_FEXT(PSC->PTSTAT, PSC_PTSTAT_GOSTAT0) == CSL_PSC_PTSTAT_GOSTAT0_IN_TRANSITION );
  
  // If we are already in that state, just return
  if ( CSL_FEXT(PSC->MDSTAT[CSL_PSC_PRU], PSC_MDSTAT_STATE) == CSL_PSC_MDSTAT_STATE_ENABLE ) return;
    
  // Perform transition
  CSL_FINST(PSC->MDCTL[CSL_PSC_PRU], PSC_MDCTL_NEXT, ENABLE);
  CSL_FINST(PSC->PTCMD, PSC_PTCMD_GO0, SET);

  // Wait for transition to complete
  while ( CSL_FEXT(PSC->PTSTAT, PSC_PTSTAT_GOSTAT0) == CSL_PSC_PTSTAT_GOSTAT0_IN_TRANSITION );
  
  // Wait and verify the state
  while ( CSL_FEXT(PSC->MDSTAT[CSL_PSC_PRU], PSC_MDSTAT_STATE) != CSL_PSC_MDSTAT_STATE_ENABLE );
}

static void LOCAL_pruPscDisable( void )
{
  CSL_PscRegsOvly PSC = (CSL_PscRegsOvly) CSL_PSC_0_REGS;
  
  // Wait for any outstanding transition to complete
  while ( CSL_FEXT(PSC->PTSTAT, PSC_PTSTAT_GOSTAT0) == CSL_PSC_PTSTAT_GOSTAT0_IN_TRANSITION );
  
  // If we are already in that state, just return
  if ( CSL_FEXT(PSC->MDSTAT[CSL_PSC_PRU], PSC_MDSTAT_STATE) == CSL_PSC_MDSTAT_STATE_SYNCRST ) return;
    
  // Perform transition
  CSL_FINST(PSC->MDCTL[CSL_PSC_PRU], PSC_MDCTL_NEXT, SYNCRST);
  CSL_FINST(PSC->PTCMD, PSC_PTCMD_GO0, SET);

  // Wait for transition to complete
  while ( CSL_FEXT(PSC->PTSTAT, PSC_PTSTAT_GOSTAT0) == CSL_PSC_PTSTAT_GOSTAT0_IN_TRANSITION );
  
  // Wait and verify the state
  while ( CSL_FEXT(PSC->MDSTAT[CSL_PSC_PRU], PSC_MDSTAT_STATE) != CSL_PSC_MDSTAT_STATE_SYNCRST );
}


/***********************************************************
* End file                                                 *
***********************************************************/




